<?php

trait CatalogProduct
{
    /**
     * getForm
     *
     * @param  mixed $langId
     * @param  mixed $productType
     * @param  mixed $recordId  - isRequested = 0 then it is product id  else request id
     * @param  mixed $isRequested - is requested catalog by seller
     * @return object
     */

    private function getCatalogForm($langId, $recordId = 0, $shippedByUserId = 0): object
    {
		$allowSale = FatApp::getConfig("CONF_ALLOW_SALE", FatUtility::VAR_INT, 0);
        $frm = new Form('frmProduct');

        $fld = $frm->addSelectBox(Labels::getLabel('FRM_LANGUAGE', $langId), 'lang_id', Language::getDropDownList(), $langId, [], '');
        $fld->requirements()->setRequired();
        if (1 > $recordId) {
            $fld->addFieldTagAttribute('disabled', 'disabled');
        }

        $frm->addHiddenField('', 'product_type', Product::PRODUCT_TYPE_PHYSICAL);
        $frm->addRequiredField(Labels::getLabel('FRM_PRODUCT_IDENTIFIER', $langId), 'product_identifier');
        $frm->addRequiredField(Labels::getLabel('FRM_PRODUCT_NAME', $langId), 'product_name');

        $fld = $frm->addSelectBox(Labels::getLabel('FRM_BRAND', $langId), 'product_brand_id', []);
        if (FatApp::getConfig("CONF_PRODUCT_BRAND_MANDATORY", FatUtility::VAR_INT, 1)) {
            $fld->requirements()->setRequired();
        }
        $fld = $frm->addSelectBox(Labels::getLabel('FRM_CATEGORY', $langId), 'ptc_prodcat_id', []);
        $fld->requirements()->setRequired();

		if ($allowSale) {
			$lbl = Labels::getLabel('FRM_MINIMUM_SELLING_PRICE', $langId);
			$fld = $frm->addFloatField($lbl . ' [' . CommonHelper::getCurrencySymbol(true) . ']', 'product_min_selling_price', '');
			$fld->requirements()->setRange('0.01', '99999999.99');
		}

        $fld = $frm->addRequiredField(Labels::getLabel('FRM_PRODUCT_WARRANTY', $langId), 'product_warranty');
		$fld->requirements()->setInt();
		$fld->requirements()->setPositive();
		$frm->addHiddenField('', 'product_warranty_unit');
        $frm->addHtmlEditor(Labels::getLabel('FRM_DESCRIPTION', $langId), 'product_description');
        $frm->addTextBox(Labels::getLabel('FRM_YOUTUBE_VIDEO_URL', $langId), 'product_youtube_video');
        $frm->addCheckBox(Labels::getLabel('FRM_MARK_AS_FEATURED', $langId), 'product_featured', 1, array(), false, 0);
        $frm->addCheckBox(Labels::getLabel("FRM_ACTIVE", $langId), 'product_active', applicationConstants::YES, array(), true, 0);
		$frm->addCheckBox(Labels::getLabel("FRM_APPROVAL_STATUS", $langId), 'product_approved', applicationConstants::YES, array(), true, 0);

		$frm->addSelectBox(Labels::getLabel('LBL_Variants/Options', $langId), 'option_groups', [], '', array('placeholder' => Labels::getLabel('LBL_Select_Varinats/Options', $langId)));
		
        $frm->addTextBox(Labels::getLabel('FRM_PRODUCT_TAGS', $langId), 'tag_name');
		
		$taxCatArr = Tax::getSaleTaxCatArr($langId);
		$frm->addSelectBox(Labels::getLabel('LBL_Tax_Category[Rent]', $langId), 'ptt_taxcat_id_rent', $taxCatArr)->requirements()->setRequired();
		
		if ($allowSale) {
            $frm->addSelectBox(Labels::getLabel('LBL_Tax_Category[Sale]', $langId), 'ptt_taxcat_id', $taxCatArr)->requirements()->setRequired();
        }
		
		$fldModel = $frm->addTextBox(Labels::getLabel('LBL_Model', $langId), 'product_model');
        if (FatApp::getConfig("CONF_PRODUCT_MODEL_MANDATORY", FatUtility::VAR_INT, 1)) {
            $fldModel->requirements()->setRequired();
        }
		
		$fld = $frm->addCheckBox(Labels::getLabel('FRM_AVAILABLE_FOR_CASH_ON_DELIVERY_(COD)', $langId), 'product_cod_enabled', 1, array(), false, 0);
		
		/*if($shippedByUserId > 0) {
			$fulfillmentType = Shop::getAttributesByUserId($userId, 'shop_fulfillment_type');
			$shopDetails = Shop::getAttributesByUserId($userId, null, false);
			$address = new Address(0, $langId);
			$addresses = $address->getData(Address::TYPE_SHOP_PICKUP, $shopDetails['shop_id']);
			$fulfillmentType = empty($addresses) ? Shipping::FULFILMENT_SHIP : $fulfillmentType;
		}*/
			
		$fulFillmentArr = Shipping::getFulFillmentArr($langId, FatApp::getConfig('CONF_FULFILLMENT_TYPE', FatUtility::VAR_INT, -1));
		$fld = $frm->addSelectBox(Labels::getLabel('FRM_FULFILLMENT_METHOD', $langId), 'product_fulfillment_type', $fulFillmentArr, applicationConstants::NO, [], Labels::getLabel('FRM_SELECT', $langId));
		$fld->requirements()->setRequired();

		if (FatApp::getConfig("CONF_PRODUCT_DIMENSIONS_ENABLE", FatUtility::VAR_INT, 1)) {
			$shipPackArr = ShippingPackage::getNames();
			$frm->addSelectBox(Labels::getLabel('FRM_SHIPPING_PACKAGE', $langId), 'product_ship_package', $shipPackArr, '', [], Labels::getLabel('FRM_SELECT', $langId))->requirements()->setRequired();
		}

		if (FatApp::getConfig("CONF_PRODUCT_WEIGHT_ENABLE", FatUtility::VAR_INT, 1)) {
			$weightUnitsArr = applicationConstants::getWeightUnitsArr($langId);
			$frm->addSelectBox(Labels::getLabel('FRM_WEIGHT_UNIT', $langId), 'product_weight_unit', $weightUnitsArr, '', [], Labels::getLabel('FRM_SELECT', $langId))->requirements()->setRequired();

			$weightFld = $frm->addFloatField(Labels::getLabel('FRM_WEIGHT', $langId), 'product_weight', '0.00');
			$weightFld->requirements()->setRequired(true);
			$weightFld->requirements()->setFloatPositive();
			$weightFld->requirements()->setRange('0.01', '9999999999');
		}
		
		$shipProfileArr = ShippingProfile::getProfileArr($langId, $shippedByUserId, true, true);
		$frm->addSelectBox(Labels::getLabel('LBL_Shipping_Profile', $langId), 'shipping_profile', $shipProfileArr, '', [], '');
			
		$fld = $frm->addSelectBox(Labels::getLabel('FRM_USER', $langId), 'product_seller_id', []);
		
		$frm->addHiddenField('', 'product_id', $recordId);
        $frm->addHiddenField('', 'temp_record_id');
        $frm->addHiddenField('', 'specifications');
        $frm->addHiddenField('', 'filespecifications');
        $frm->addHiddenField('', 'num_attributes');
        $frm->addHiddenField('', 'text_attributes');
        $frm->addSubmitButton('', 'btn_submit', Labels::getLabel('LBL_Save', $langId));
        return $frm;
    }
	
	public function singleForm($productId = 0, $prodCatId = 0)
    {
        $this->objPrivilege->canEditProducts();
        $productId = FatUtility::int($productId);
        $productType = Product::PRODUCT_TYPE_PHYSICAL;
		$totalProducts = $shippedByUserId = 0;
		
		$langId = FatApp::getPostedData('langId', FatUtility::VAR_INT, 0);
        if (1 > $langId) {
            $langId = $this->adminLangId;
        }
		
		if ($productId > 0) {
			$productData = $this->getProductData($productId, $langId);
			$shippedByUserId = $productData['product_seller_id'];
			if (FatApp::getConfig('CONF_SHIPPED_BY_ADMIN_ONLY', FatUtility::VAR_INT, 0)) {
				$shippedByUserId = 0;
			}
		}
		
		$frm = $this->getCatalogForm($langId, $productId, $shippedByUserId);
		$imgFrm = $this->getImageForm($langId);
		
		$productTags = $productOptions = $prodOptValCombArr = [];

		if ($productId > 0) {
			if(!empty($productData['brand_name'])) {
				$fld = $frm->getField('product_brand_id');
				$fld->options = [$productData['product_brand_id'] => $productData['brand_name']];
			}
			
			if(!empty($productData['category_name'])) {
				$fld = $frm->getField('ptc_prodcat_id');
				$fld->options = [$productData['ptc_prodcat_id'] => $productData['category_name']];
			}
			
			$prodCatId = $productData['ptc_prodcat_id'];
			
			$fld = $frm->getField('product_seller_id');
            if ($productData['product_seller_id'] > 0) {
                $userShopName = User::getUserShopName($productData['product_seller_id'], $langId, false);
                $fld->options = [$productData['product_seller_id'] => $userShopName['user_name'] . ' || ' . $userShopName['shop_name']];
            }
			$totalProducts = Product::getCatalogProductCount($productId);
			$productOptions = Product::getProductOptions($productId, $langId, true);
			$prodOptValCombArr = CommonHelper::combinationOfElementsOfArr($productOptions, 'optionValues', ' | ');

			$imgProductIdFld = $imgFrm->getField('product_id');
			$imgProductIdFld->value = $productId;

		} else {
            $tempProductId = time() . AdminAuthentication::getLoggedAdminId();
            $productData = [
                'temp_record_id' => $tempProductId,
                'product_id' => 0
            ];
        }
		
		$productData['lang_id'] = $langId;

		$frm->fill($productData);
		
		$this->set("frm", $frm);
		$this->set("imgFrm", $imgFrm);
		$this->set('productOptions', $productOptions);
		$this->set('prodOptValCombArr', $prodOptValCombArr);
        $this->set('productId', $productId);
        $this->set('productTags', $productTags);
        $this->set('prodCatId', $prodCatId);
        $this->set('productType', $productType);
        $this->set('totalProducts', $totalProducts);
        $this->set('productData', $productData);
        
        $this->set("includeEditor", true);

        $isCustomFields = false;
        if ($productId > 0) {
            $prod = new Product();
            $productCategories = $prod->getProductCategories($productId);
            if (!empty($productCategories)) {
                $selectedCat = array_keys($productCategories);
                $pcObj = new ProductCategory($selectedCat[0]);
                $isCustomFields = $pcObj->isCategoryHasCustomFields($langId);
            }
        }

		$paymentMethod = new PaymentMethods();

        $this->set('isCustomFields', $isCustomFields);
        $this->set('lang_id', $langId);
        $this->set('codEnabled', $paymentMethod->cashOnDeliveryIsActive());
		$this->set('allowForSale', FatApp::getConfig("CONF_ALLOW_SALE", FatUtility::VAR_INT, 0));
		$this->set('imageSize', imagesSizes::productImageSizeArr()[applicationConstants::getActiveTheme()]);
		$this->set('formLayout', Language::getLayoutDirection($langId));
		
		if (FatUtility::isAjaxCall()) {
            $this->set('html', $this->_template->render(false, false, 'products/single-catalog-form/single-form.php', true));
            //$this->set('html', 'Hello my friends');
            $this->_template->render(false, false, 'json-success.php', true, false);
            return;
        }
		
		$this->_template->addJs(array('js/cropper.js', 'js/cropper-main.js', 'js/jquery-sortable-lists.js', 'js/tagify.min.js', 'js/tagify.polyfills.min.js', 'products/page-js/single-form.js', 'js/select2.js'));
        $this->_template->addCss(array('css/cropper.css', 'css/tagify.css', 'css/select2.min.css'));
		
        $this->_template->render(true, true, 'products/single-catalog-form/single-form.php');
    }
	
	public function getProductData($productId, $langId) 
	{
		$prodData = Product::getAttributesById($productId);
		$prod = new Product();
		$productLangData = $prod->getAttributesByLangId($langId, $productId);
		if (!empty($productLangData)) {
			$prodData = array_merge($prodData, $productLangData);
		}
		
		$taxData = array();
		$tax = Tax::getTaxCatObjByProductId($productId, $langId);
		if ($prodData && $prodData['product_seller_id'] > 0) {
			$tax->addCondition('ptt_seller_user_id', '=', $prodData['product_seller_id']);
		} else {
			$tax->addCondition('ptt_seller_user_id', '=', 0);
		}

		$activatedTaxServiceId = Tax::getActivatedServiceId();

		$tax->addFld(['ptt_taxcat_id', 'ptt_taxcat_id_rent']);
		if ($activatedTaxServiceId) {
			$tax->addFld(array('concat(IFNULL(t_l.taxcat_name,t.taxcat_identifier), " (",t.taxcat_code,")")as taxcat_name', 'concat(IFNULL(t_lrent.taxcat_name,trent.taxcat_identifier), " (",trent.taxcat_code,")")as taxcat_name_rent'));
		} else {
			$tax->addFld(array('IFNULL(t_l.taxcat_name,t.taxcat_identifier)as taxcat_name', 'IFNULL(t_lrent.taxcat_name,trent.taxcat_identifier)as taxcat_name_rent'));
		}

		$tax->doNotCalculateRecords();
		$tax->setPageSize(1);
		$tax->addOrder('ptt_seller_user_id', 'ASC');

		$rs = $tax->getResultSet();
		$taxData = FatApp::getDb()->fetch($rs);

		if (!empty($taxData)) {
			$prodData['ptt_taxcat_id'] = $taxData['ptt_taxcat_id'];
			$prodData['ptt_taxcat_id_rent'] = $taxData['ptt_taxcat_id_rent'];
		}
		
		$prodSpecificsDetails = Product::getProductSpecificsDetails($productId);
        $prodData['product_warranty'] = isset($prodSpecificsDetails['product_warranty']) ? $prodSpecificsDetails['product_warranty'] : '';
		
		$shippedByUserId = $prodData['product_seller_id'];
        if (FatApp::getConfig('CONF_SHIPPED_BY_ADMIN_ONLY', FatUtility::VAR_INT, 0)) {
            $shippedByUserId = 0;
        }
		
		$profSrch = ShippingProfileProduct::getSearchObject();
        $profSrch->addCondition('shippro_product_id', '=', $productId);
        $profSrch->addCondition('shippro_user_id', '=', $shippedByUserId);
        $proRs = $profSrch->getResultSet();
        $profileData = FatApp::getDb()->fetch($proRs);
        if (!empty($profileData)) {
            $prodData['shipping_profile'] = $profileData['profile_id'];
        }
		
		$brandData = $this->getBrandDetails($productId, $langId);
		if (!empty($brandData)) {
			$prodData['brand_name'] = $brandData[Brand::tblFld('name')] ?? $brandData[Brand::tblFld('identifier')];
		}
		
		$categoryData = $this->getCategoryDetails($productId, $langId);
		$prodData = array_merge($prodData, $categoryData);
		
		$productTags = Product::getProductTags($productId);
		
		if(!empty($productTags)) {
			$tagData = array();
			foreach ($productTags as $key => $data) {
				$tagData[$key]['id'] = $data['tag_id'];
				$tagData[$key]['value'] = $data['tag_identifier'];
			}
			
			if(!empty($tagData)) {
				//$prodData['tag_name'] = htmlspecialchars(json_encode($tagData), ENT_QUOTES, 'UTF-8');
				$prodData['tag_name'] = json_encode($tagData);
			}
		}
		
		$pObj = new Product($productId);
        $isOptWithSizeChart = $pObj->checkOptionWithSizeChart();

        $productSizeChart = [];
        if ($isOptWithSizeChart) {
            $productSizeChart = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $productId, 0, $langId, false, 0, 0, true);
        }

        $prodData['sizeChartArr'] = $productSizeChart;
		
		return $prodData;
	}
	
	private function getCategoryDetails($productId, $langId)
	{
		$prod = new Product();
		$productCategories = $prod->getProductCategories($productId);
		$prodCatData = [];
		if (!empty($productCategories)) {
			$selectedCat = array_keys($productCategories);
			$prodCat = new ProductCategory();
			$prodCatData = [
				'ptc_prodcat_id' => $selectedCat[0],
				'category_name' => html_entity_decode($prodCat->getParentTreeStructure($selectedCat[0], 0, '', $langId)),
			];
		}
		return $prodCatData;
	}
	
	private function getBrandDetails($productId, $langId)
	{
		$srch = Product::getSearchObject($langId);
		$srch->joinTable(Brand::DB_TBL, 'INNER JOIN', 'tp.product_brand_id = brand.brand_id', 'brand');
		$srch->joinTable(Brand::DB_TBL_LANG, 'LEFT OUTER JOIN', 'brandlang_brand_id = brand.brand_id AND brandlang_lang_id = ' . $langId);
		$srch->addMultipleFields(array('product_brand_id', 'IFNULL(brand_name,brand_identifier) as brand_name', 'IFNULL(brand.brand_active,1) AS brand_active', 'IFNULL(brand.brand_deleted,0) AS brand_deleted'));
		$srch->addCondition('product_id', '=', $productId);
		$srch->addHaving('brand_active', '=', applicationConstants::YES);
		$srch->addHaving('brand_deleted', '=', applicationConstants::NO);
		$rs = $srch->getResultSet();
		return FatApp::getDb()->fetch($rs);
	}
	
	public function setupProductSingleForm()
	{
		$this->objPrivilege->canEditProducts();
        $productId = FatApp::getPostedData('product_id', FatUtility::VAR_INT, 0);
        $langId = FatApp::getPostedData('lang_id', FatUtility::VAR_INT, 0);
        $prodCatId = FatApp::getPostedData('ptc_prodcat_id', FatUtility::VAR_INT, 0);
        $prodBrandId = FatApp::getPostedData('product_brand_id', FatUtility::VAR_INT, 0);
        $shippedByUserId = FatApp::getPostedData('product_seller_id', FatUtility::VAR_INT, 0);
        $sizeChartTempId = FatApp::getPostedData('size_chart_temp_id', FatUtility::VAR_INT, 0);
		$optionGroups = FatApp::getPostedData('prod_options');
		$flagToRemoveAllOptions = false;
		
		/*if($productId > 0 && FatApp::getConfig('CONF_SHIPPED_BY_ADMIN_ONLY', FatUtility::VAR_INT, 0) == 0) {
			$productData = Product::getAttributesById($productId);
			$shippedByUserId = $productData['product_seller_id'];
		}*/
		
        $frm = $this->getCatalogForm($langId, $productId, $shippedByUserId);
		
		$postedData = FatApp::getPostedData();
        $post = $frm->getFormDataFromArray(FatApp::getPostedData());

        if (false === $post) {
            Message::addErrorMessage(current($frm->getValidationErrors()));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		if(empty($optionGroups)) {
			$optionGroups = [];
		}
		
		if($productId > 0) {
			$productOptions = Product::getProductOptions($productId, $langId);
			$attachedOptIds = array_column($productOptions, 'option_id');
			$results = array_diff($optionGroups, $attachedOptIds);
			
			if(count($optionGroups) != count($attachedOptIds) || !empty($results)) {
				$optionSrch = SellerProduct::getSearchObject();
				$optionSrch->addCondition('selprod_product_id', '=', $productId);
				$optionSrch->addCondition('selprod_deleted', '=', applicationConstants::NO);

				$rs = $optionSrch->getResultSet();
				$row = FatApp::getDb()->fetch($rs);

				if (!empty($row)) {
					FatUtility::dieJsonError(Labels::getLabel('MSG_Inventory_already_added_for_this_product_you_can_not_update_option(s)', $langId));
					return;
				}
				$flagToRemoveAllOptions = true;
			}
		}
			
		$sizeChartAttach = false;	
		if(!empty($optionGroups)) {
			$optionHasSepImage = false;
			foreach($optionGroups as $optionGroup) {
				$optionDetail = Option::getAttributesById($optionGroup, ['option_is_separate_images', 'option_attach_sizechart']);
				
				if(!empty($optionDetail) && $optionDetail['option_attach_sizechart'] == 1) {
					$sizeChartAttach = true;
				}
				
				if($optionHasSepImage == true && !empty($optionDetail) && $optionDetail['option_is_separate_images'] == 1) {
					Message::addErrorMessage(Labels::getLabel('LBL_you_can_not_attach_more_than_one_option_having_separate_image', $langId));
					FatUtility::dieJsonError(Message::getHtml());
				} else if(!empty($optionDetail) && $optionDetail['option_is_separate_images'] == 1) {
					$optionHasSepImage = true;
				}
			}
		}

        if ($prodBrandId < 1 && FatApp::getConfig("CONF_PRODUCT_BRAND_MANDATORY", FatUtility::VAR_INT, 1)) {
            Message::addErrorMessage(Labels::getLabel('MSG_Please_Choose_Brand_From_List', $langId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		if ($prodCatId < 1) {
            Message::addErrorMessage(Labels::getLabel('MSG_Please_Choose_Category_From_List', $langId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		if (!FatApp::getConfig("CONF_ALLOW_SALE", FatUtility::VAR_INT, 0) && $productId > 1) {
            $minSellingPrice = Product::getAttributesById($productId, 'product_min_selling_price');
			if (!empty($minSellingPrice)) {
				$post['product_min_selling_price'] = $minSellingPrice;
			}
        }
		
		$post['product_brand_id'] = $prodBrandId;
		$post['product_seller_id'] = $shippedByUserId;
		$prod = new Product($productId);
        if (!$prod->saveProductData($post)) {
            Message::addErrorMessage($prod->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
        Product::updateMinPrices($productId);

		$prodLangdata['product_name'][$langId] = $post['product_name'];
		$prodLangdata['product_youtube_video'][$langId] = $post['product_youtube_video'];
		$prodLangdata['product_description_'.$langId] = $post['product_description'];
		
        if (!$prod->saveProductLangData($prodLangdata)) {
            Message::addErrorMessage($prod->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        if (!$prod->saveProductCategory($prodCatId)) {
            Message::addErrorMessage($prod->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		$saleTaxCat = 0;
		if(!empty($post['ptt_taxcat_id'])) {
			$saleTaxCat = $post['ptt_taxcat_id'];
		}

		if (!$prod->saveProductTax($saleTaxCat, $shippedByUserId, SellerProduct::PRODUCT_TYPE_PRODUCT, $post['ptt_taxcat_id_rent'])) {
			Message::addErrorMessage($prod->getError());
			FatUtility::dieJsonError(Message::getHtml());
		}
		
		
		$this->setupTags($prod, $post['tag_name'], $langId);
		
		if (isset($post['shipping_profile']) && $post['shipping_profile'] > 0) {
            $shipProProdData = array(
                'shippro_shipprofile_id' => $post['shipping_profile'],
                'shippro_product_id' => $prod->getMainTableRecordId(),
                'shippro_user_id' => $shippedByUserId,
            );

            $spObj = new ShippingProfileProduct();
            if (!$spObj->addProduct($shipProProdData)) {
                Message::addErrorMessage($spObj->getError());
                FatUtility::dieJsonError(Message::getHtml());
            }
        }
		
		if(!empty($postedData['specifications'])) {
			$this->addUpdateSpecifcations($prod, $langId, $productId, $postedData['specifications']);
		}
		
		if(!empty($postedData['filespecifications'])) {
			$response = $this->addUpdateFileSpecifications($prod, $post['filespecifications'], $langId, 0);
		}
		
		if($flagToRemoveAllOptions == true) {
			if (!$prod->removeProductAllOption()) {
				FatUtility::dieJsonError($prod->getError());
			}
		}
			
		if(!empty($optionGroups)) {
			foreach($optionGroups as $optionGroup) {
				if (!$prod->addUpdateProductOption($optionGroup)) {
					FatUtility::dieJsonError($prod->getError());
				}
			}
		}
		$dataToSave = [
			'ps_product_id' => $prod->getMainTableRecordId(),
			'product_warranty' => $post['product_warranty'],
		];
		$productSpecifics = new ProductSpecifics($prod->getMainTableRecordId());
        $productSpecifics->assignValues($dataToSave);
        if (!$productSpecifics->addNew(array(), $dataToSave)) {
            Message::addErrorMessage($productSpecifics->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		if($sizeChartTempId > 0 && $sizeChartAttach == true) {
			$criteria = array('afile_id' => $sizeChartTempId);
			$this->moveTempFiles($prod->getMainTableRecordId(), $criteria, '', '', true);
		}
		
		if($sizeChartAttach == false) {
			$fileHandlerObj = new AttachedFile();
			$fileHandlerObj->deleteFile(AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $prod->getMainTableRecordId());
		}
		
		//@todo Size chart can be handle with below code, we can pass record id in criteria instead of any file id or type
		$criteria = array('afile_type' => AttachedFile::FILETYPE_PRODUCT_IMAGE_TEMP);
		$this->moveTempFiles($prod->getMainTableRecordId(), $criteria, AttachedFile::FILETYPE_PRODUCT_IMAGE);
		
		$this->setupProdCatCustomFieldsData($prod, $post, $langId);
		
		$newTabLangId = 0;
		$siteDefaultLangId = FatApp::getConfig('conf_default_site_lang', FatUtility::VAR_INT, 1);
        
        $languages = Language::getAllNames();
        foreach ($languages as $languageId => $langName) {
			if($productId == 0) {
				if($languageId != $siteDefaultLangId) {
					$newTabLangId = $languageId;
					break;
				}
			} else {
				if (!$row = Product::getAttributesByLangId($languageId, $prod->getMainTableRecordId())) {
					$newTabLangId = $languageId;
					break;
				}
			}
        }
		
		$response = [
			'msg' => Labels::getLabel('LBL_Product_Setup_Successful', $langId),
			'productId' => $prod->getMainTableRecordId(),
			'langId' => $newTabLangId,
			'status' => true,
		];
		
		echo json_encode($response);
		
	}
	
	private function addUpdateSpecifcations($prodObj, $langId, $productId = 0, $specArr = [])
	{
		if($productId == 0 && !empty($specArr)) {
			foreach($specArr as $prodSpecDetail) {
				$prodSpecId = $prodSpecDetail['id'];
				$post['prodspec_identifier'] = $prodSpecDetail['identifier'];
				if (!$specId = $prodObj->saveProductSpecifications($prodSpecId, $langId, $prodSpecDetail['name'], $prodSpecDetail['value'], $prodSpecDetail['group'], 0, false, 0, $post)) {
					Message::addErrorMessage($prodObj->getError());
					FatUtility::dieJsonError(Message::getHtml());
				}
			}
		} else if($productId > 0 && !empty($specArr)) {
			$productObj = new Product($productId);
			$productSpecifications = $productObj->getProdSpecificationsByLangId($langId);
			
			if(!empty($productSpecifications)) {
				$siteDefaultLangId = FatApp::getConfig('conf_default_site_lang', FatUtility::VAR_INT, 1);
				/*$newSpecAdded = false;
				$existingSpecArr = $newSpecArr = [];
				foreach($specArr as $specDetail) {
					if($specDetail['id'] == 0) {
						$newSpecAdded = true;
						$newSpecArr[] =  [
							'prodspec_id' => 0,
							'prodspec_identifier' => $specDetail['identifier'],
							'prodspec_name' => $specDetail['name'],
							'prodspec_value' => $specDetail['value'],
							'prodspec_group' => $specDetail['group'],
						];
					} else {
						$existingSpecArr[] =  [
							'prodspec_id' => $specDetail['id'],
							'prodspec_identifier' => $specDetail['identifier'],
							'prodspec_name' => $specDetail['name'],
							'prodspec_value' => $specDetail['value'],
							'prodspec_group' => $specDetail['group'],
						];
					}
				}*/
				
				$specIdArr = array_column($productSpecifications, 'prodspec_id');
				$postedSpecIdArr = array_column($specArr, 'id');
				
				$arrDiff = array_diff($specIdArr, $postedSpecIdArr);
				
				if(!empty($arrDiff)) {
					foreach($arrDiff as $prodSpecId) {
						$prodSpec = new ProdSpecification($prodSpecId);
						if (!$prodSpec->deleteRecord(true)) {
							Message::addErrorMessage($prodSpec->getError());
							FatUtility::dieJsonError(Message::getHtml());
						}

						/* [ DELETE UPLOADED FILE */
						$this->deleteSpecFile($productId, $prodSpecId);
					}
				}
			}	
				
			foreach($specArr as $prodSpecDetail) {
				$prodSpecId = $prodSpecDetail['id'];
				$post['prodspec_identifier'] = $prodSpecDetail['identifier'];

				if(($prodSpecDetail['name'] == '' || $prodSpecDetail['value'] == '') && $siteDefaultLangId != $langId) {
					continue;
				}
				if (!$specId = $prodObj->saveProductSpecifications($prodSpecId, $langId, $prodSpecDetail['name'], $prodSpecDetail['value'], $prodSpecDetail['group'], 0, false, 0, $post)) {
					Message::addErrorMessage($prodObj->getError());
					FatUtility::dieJsonError(Message::getHtml());
				}
			}
		}
		
		return true;
	}
	
	public function addUpdateFileSpecifications ($prodObj, $fileSpecifications, $langId, $autoUpdateOtherLangsData = 0)
    {
        if (empty($fileSpecifications)) {
            return true;
        }
        
		
        foreach ($fileSpecifications as $specification) {
            $data['prodspec_identifier'] = $specification['identifier'];
            $data['prodspec_name'] = $specification['name'];
            $data['prodspec_value'] = '';
            $data['prodspec_group'] = '';
            if (!$prodObj->saveScfProductSpecifications($specification['id'], $langId, $data, applicationConstants::YES)) {
                Message::addErrorMessage($prodObj->getError());
                return false;
            }

            $specId = $prodObj->prodSpecId;
            $criteria = array('afile_id' => $specification['temp_file_id']);
            $this->moveTempFiles($prodObj->getMainTableRecordId(), $criteria, AttachedFile::FILETYPE_PRODUCT_SPECIFICATION_FILE, $specId);

            if (1 === $autoUpdateOtherLangsData) {
                /*$languages = Language::getAllNames(false);
                unset($languages[$langId]);
                $success = true;
                foreach ($languages as $toLangId => $langData) {
                    $translateLangobj = new TranslateLangData(ProdSpecification::DB_TBL);
                    $translatedData = $translateLangobj->directTranslate($specification, $toLangId);
                    if (isset($translatedData[$toLangId]) && !empty($translatedData[$toLangId])) {
                        $translatedData = $translatedData[$toLangId];
                        $languageData['prodspec_name'] = $translatedData['name'];
                        if (!$prodObj->saveScfProductSpecifications($specId, $toLangId, $languageData)) {
                            Message::addErrorMessage($prodObj->getError());
                            $success = false;
                            break;
                        }
                    }
                }
                if (true != $success) {
                    return false;
                }*/
            }
        }
        return true;
    }
	
	private function setupTags($prodObj, $tags, $langId)
    {
        if (!empty($tags)) {
            $productTags = json_decode($tags, true);
            foreach ($productTags as $tag) {
                if (!isset($tag['id'])) {
                    $tagObj = new Tag();
                    $tagObj->assignValues([
                        'tag_identifier' => $tag['value'],
                        'tag_user_id' => 0
                    ]);
                    if (!$tagObj->save()) {
                        LibHelper::exitWithError($tagObj->getError(), true);
                    }
                    $tagId = $tagObj->getMainTableRecordId();

                    $data['tag_name'] = $tag['value'];
                    if (!$tagObj->updateLangData($langId, $data)) {
                        LibHelper::exitWithError($tagObj->getError(), true);
                    }
                } else {
                    $tagId = $tag['id'];
                }

                if (!$prodObj->addUpdateProductTag($tagId)) {
                    LibHelper::exitWithError($prodObj->getError(), true);
                }

                /* update product tags association and tag string in products lang table[ */
                Tag::updateTagStrings($tagId);
                /* ] */
                
            }
        }
        return true;
    }
	
	public function validateAttachedOptions()
	{
		$post = FatApp::getPostedData();
		
		/*if(empty($post['prod_options'])) {
			FatUtility::dieJsonError(Labels::getLabel('LBL_Invalid_Request', $this->adminLangId));
		}*/
		
		$sizeChartCanBeAttached = $optionHasSepImage = 0;
		$optionValArr = [];
		
		if(!empty($post['prod_options'])) {
			$optionsDetail = Option::getOptionsData($post['prod_options']);
			if(!empty($optionsDetail)) {
				foreach($optionsDetail as $optionDetail) {
					if($optionDetail['option_is_separate_images'] == 1) {
						$optionHasSepImage += 1;
					}
					
					if($optionDetail['option_attach_sizechart'] == 1) {
						$sizeChartCanBeAttached = 1;
					}
				}
			}
		}
		
		if($optionHasSepImage > 1) {
			FatUtility::dieJsonError(Labels::getLabel("MSG_You_can_not_attach_more_than_one_option_having_separate_image", $this->adminLangId));
		}
		
		if(!empty($post['prod_options'])) {
			$optValObj = new OptionValue();
			foreach($post['prod_options'] as $optionId) {
				$optionValues = $optValObj->getAtttibutesByOptionId($optionId, null, $this->adminLangId);
				if(!empty($optionValues)) {
					foreach($optionValues as $optionVal) {
						$optionValName = $optionVal['optionvalue_identifier'];
						if(!empty($optionVal['optionvalue_name'])) {
							$optionValName = $optionVal['optionvalue_name'];
						}
						$optionValArr[$optionId]['optionValues'][$optionVal['optionvalue_id']] = $optionValName;
					}
				}
			}
		}
		
		if(!empty($optionValArr)) {
			$optionValArr = CommonHelper::combinationOfElementsOfArr($optionValArr, 'optionValues', ' | ');
		}

        $this->set('optionValArr', $optionValArr);
        $this->set('sizeChartCanBeAttached', $sizeChartCanBeAttached);
        $this->set('msg', Labels::getLabel('LBL_Option_Updated_Successfully', $this->adminLangId));
        $this->_template->render(false, false, 'json-success.php');
	}
	
	public function setupSpecFiles()
	{
		$isFile = 1;
		$post = FatApp::getPostedData();
		if (empty($post)) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request_Or_File_not_supported', $this->adminLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		$productId = FatUtility::int($post['product_id']);
        $tempProductId = FatUtility::int($post['temp_record_id']);
        $langId = FatUtility::int($post['langId']);
		if (1 > $productId) {
            $productId = $tempProductId;
        }
		
		if ($_FILES['file']['size'] > 10240000) {
			Message::addErrorMessage(Labels::getLabel('MSG_Please_upload_file_size_less_than_10MB', $this->adminLangId));
			FatUtility::dieJsonError(Message::getHtml());
		}
		
		$fileExt = pathinfo($_FILES['file']['name'], PATHINFO_EXTENSION);
		$fileExt = strtolower($fileExt);
		if (in_array($fileExt, applicationConstants::allowedImageFileExtensions())) {
			$isFile = 0;
		}
			
		$prodSpecId = 0;
		$fileHandlerObj = new AttachedFileTemp();
		 
		if (!$res = $fileHandlerObj->saveAttachment($_FILES['file']['tmp_name'], AttachedFile::FILETYPE_PRODUCT_SPECIFICATION_FILE_TEMP, $productId, $prodSpecId, $_FILES['file']['name'], -1, false, $langId, 0, 0, true)) {
			Message::addErrorMessage($fileHandlerObj->getError());
			FatUtility::dieJsonError(Message::getHtml());
		}
		
		if($isFile == 0) {
			$retFileTag = '<img class="img-thumbnail image-small mt-2" src="' . UrlHelper::generateUrl('Image', 'attachment', [$fileHandlerObj->getMainTableRecordId(), applicationConstants::YES, 50,50], '/') . '">';
		} else {
			$retFileTag = '<a class="btn btn-link" target="_blank" href="' . UrlHelper::generateUrl('Image', 'attachment', [$fileHandlerObj->getMainTableRecordId(), applicationConstants::YES], '/') . '"><i class="fa fa-download" aria-hidden="true"> &nbsp;&nbsp;'.$_FILES['file']['name'].'</i></a>';
		}
		
		$this->set("file_htm_tag", $retFileTag);
        $this->set("spec_temp_id", $fileHandlerObj->getMainTableRecordId());
        $this->set("prodspec_is_file", $isFile);
        $this->set("msg", Labels::getLabel('MSG_FILE_UPLOADED_SUCCESSFULLY', $this->adminLangId));
        $this->_template->render(false, false, 'json-success.php', true, false);
	}
	
	public function autoCompleteUserShopName()
    {
		$page = FatApp::getPostedData('page', FatUtility::VAR_INT, 1);
        if ($page < 2) {
            $page = 1;
        }
        $pagesize = FatApp::getConfig('CONF_PAGE_SIZE');
        $post = FatApp::getPostedData();
        $srch = new SearchBase(User::DB_TBL, 'tu');
        $srch->joinTable('tbl_shops', 'INNER JOIN', 'tu.user_id=ts.shop_user_id', 'ts');
        $srch->addOrder('user_name', 'asc');
        if (!empty($post['keyword'])) {
            $cnd = $srch->addCondition('user_name', 'LIKE', '%' . $post['keyword'] . '%');
            $cnd = $cnd->attachCondition('shop_identifier', 'LIKE', '%' . $post['keyword'] . '%', 'OR');
        }
        $srch->joinTable(User::DB_TBL_CRED, 'LEFT OUTER JOIN', 'uc.' . User::DB_TBL_CRED_PREFIX . 'user_id = tu.user_id', 'uc');
        $srch->addCondition('uc.' . User::DB_TBL_CRED_PREFIX . 'active', '=', 1);
		$srch->addCondition('tu.' . User::DB_TBL_PREFIX . 'deleted', '=', 0);
        $srch->addCondition('user_is_supplier', '=', 1);

        $srch->addMultipleFields(array('user_id', 'user_name', 'shop_identifier'));
		$srch->setPageNumber($page);
        $srch->setPageSize($pagesize);
			
        $db = FatApp::getDb();
        $rs = $srch->getResultSet();
        $users = FatApp::getDb()->fetchAll($rs);
		$json = array(
            'pageCount' => $srch->pages()
        );
		
        foreach ($users as $key => $option) {
			$shopName = $option['user_name'].' || '.$option['shop_identifier'];
            $json['results'][] = array(
                'id' => $option['user_id'],
                'text' => strip_tags(html_entity_decode($shopName, ENT_QUOTES, 'UTF-8'))
            );
        }
        die(json_encode($json));
    }
	
	public function getShippingProfileOptions()
    {
        $userId = FatApp::getPostedData('userId', FatUtility::VAR_INT, 0);
        $langId = FatApp::getPostedData('langId', FatUtility::VAR_INT, 0);

        if (1 > $langId) {
			Message::addErrorMessage(Labels::getLabel('LBL_Lang_id_is_required', $this->adminLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
        $shipProfileArr = ShippingProfile::getProfileArr($langId, $userId, true, true);
		
		$fulfillmentType = -1;
		if($userId > 0) {
			$fulfillmentType = Shop::getAttributesByUserId($userId, 'shop_fulfillment_type');
			$shopDetails = Shop::getAttributesByUserId($userId, null, false);
			$address = new Address(0, $langId);
			$addresses = $address->getData(Address::TYPE_SHOP_PICKUP, $shopDetails['shop_id']);
			$fulfillmentType = empty($addresses) ? Shipping::FULFILMENT_SHIP : $fulfillmentType;
		}
		$fulFillmentArr = Shipping::getFulFillmentArr($langId, $fulfillmentType);
		$fulFillmentArr = [0 => Labels::getLabel('LBL_SELECT', $langId)] + $fulFillmentArr;
        FatUtility::dieJsonSuccess(['shipProfileArr' => $shipProfileArr, 'fullfilmentOptions' => $fulFillmentArr]);
    }
	
	public function setupSizeChartImages()
    {
        $this->objPrivilege->canEditProducts();
        $post = FatApp::getPostedData();
        if (empty($post)) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request_Or_File_not_supported', $this->adminLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		$productId = $recordId = FatUtility::int($post['product_id']);
        $tempProductId = FatUtility::int($post['temp_record_id']);
		$langId = FatUtility::int($post['lang_id']);
		
		if ($_FILES['file']['size'] > 10240000) {
			Message::addErrorMessage(Labels::getLabel('MSG_Please_upload_file_size_less_than_10MB', $this->adminLangId));
			FatUtility::dieJsonError(Message::getHtml());
		}
		
		if (1 > $productId) {
            $recordId = $tempProductId;
        }
		
		if (1 > $productId) {
			$fileHandlerObj = new AttachedFileTemp();
			$tempRecord = applicationConstants::YES;
		} else {
			$fileHandlerObj = new AttachedFile();
			$tempRecord = applicationConstants::NO;
		}
		
		/* [  DELETE OLD SIZE CHART */
        $productSizeChart = $fileHandlerObj::getMultipleAttachments(AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $recordId, 0, $langId, false, 0, 0, true);
		
        if (!empty($productSizeChart)) {
			
            foreach ($productSizeChart as $fileData) {
				if (1 > $productId) {
					if (!$fileHandlerObj->deleteTempFile(AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $recordId, 0, $langId)) {
						Message::addErrorMessage($fileHandlerObj->getError());
						FatUtility::dieJsonError(Message::getHtml());
					}
				} else {
					if (!$fileHandlerObj->deleteFile(AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $recordId, $fileData['afile_id'])) {
						Message::addErrorMessage($fileHandlerObj->getError());
						FatUtility::dieJsonError(Message::getHtml());
					}
				}
            }
        }
        /* ] */
		
		if (!$res = $fileHandlerObj->saveAttachment($_FILES['file']['tmp_name'], AttachedFile::FILETYPE_PRODUCT_SIZE_CHART, $recordId, 0, $_FILES['file']['name'], -1, false, $langId)) {
			Message::addErrorMessage($fileHandlerObj->getError());
			FatUtility::dieJsonError(Message::getHtml());
		}
		
		if (1 > $productId) {
			$sizeCharImgUrl = UrlHelper::generateUrl('Image', 'attachment', [$fileHandlerObj->getMainTableRecordId(), $tempRecord], CONF_WEBROOT_FRONTEND);
		} else {
			$sizeCharImgUrl = CommonHelper::generateUrl('image', 'productSizeChart', array($productId, "THUMB", $fileHandlerObj->getMainTableRecordId()), CONF_WEBROOT_FRONTEND);
		}
		
		$tpl = new FatTemplate('', '');
		$tpl->set('recordId', $fileHandlerObj->getMainTableRecordId());
		$tpl->set('langId', $this->adminLangId);
		$tpl->set('imgUrl', $sizeCharImgUrl);
		$sizeChartLayout = $tpl->render(false, false, 'products/single-catalog-form/image.php', true, true);
		
		$sizeChartTempId = 0;
		if($tempRecord) {
			$sizeChartTempId = $fileHandlerObj->getMainTableRecordId();
		}
		
        $this->set("size_chart_temp_id", $sizeChartTempId);
        $this->set("retFileTag", $sizeChartLayout);
        $this->set("msg", Labels::getLabel('MSG_FILE_UPLOADED_SUCCESSFULLY', $this->adminLangId));
        $this->_template->render(false, false, 'json-success.php', true, false);
		
    }
	
	public function test()
	{
		$criteria = array('afile_id' => 118);
		$this->moveTempFiles(179, $criteria, '', '', true);
		//$this->moveTempFiles(48, 152);
	}
	
	private function moveTempFiles($productId, $criteria = [], $newFileType = '', $subRecordId = '', $updateForGlobalLang = false) 
	{
        if(empty($criteria)) {
			return true;
		}
		
        $tempFiles = AttachedFileTemp::getTempImagesWithCriteria($criteria);
	
        if (!empty($tempFiles)) {
            foreach ($tempFiles as $tempFile) {
				
				/*$whr = ['smt' => 'afile_record_id = ? AND afile_type = ?', 'vals' => array($productId, $tempFile['afile_type'])];
				FatApp::getDb()->deleteRecords(AttachedFile::DB_TBL, $whr);*/
				
				$afileId = $tempFile['afile_id'];	
                unset($tempFile['afile_id']);
                $tempFile['afile_record_id'] = $productId;
                $tempFile['afile_updated_at'] = date('Y-m-d H:i:s');
				if($newFileType != '') {
					$tempFile['afile_type'] = $newFileType;
				}
				
				if($subRecordId != '') {
					$tempFile['afile_record_subid'] = $subRecordId;
				}
				
                $fileHandler = new AttachedFile();
                $fileHandler->assignValues($tempFile);
                if (!$fileHandler->save()) {
                    Message::addErrorMessage($fileHandler->getError());
                    return false;
                }
				
				if($updateForGlobalLang == true) {
					$fileHandler = new AttachedFile();
					$tempFile['afile_lang_id'] = 0;
					$fileHandler->assignValues($tempFile);
					if (!$fileHandler->save()) {
						Message::addErrorMessage($fileHandler->getError());
						return false;
					}
				}

                $whr = ['smt' => 'afile_id = ?', 'vals' => array($afileId)];
                FatApp::getDb()->deleteRecords(AttachedFileTemp::DB_TBL, $whr);
            }
        }
    }
	
	public function imageForm()
	{
		$post = FatApp::getPostedData();
		$langId = FatApp::getPostedData('langId', FatUtility::VAR_INT, 0);
        if (1 > $langId) {
            $langId = $this->adminLangId;
        }
		
		$prodOptions = $optionValAssArr = [];
		if(!empty($post['prodOptions'])) {
			$prodOptions = json_decode(stripslashes($post['prodOptions']));
			if(!empty($prodOptions)) {
			$optionsDetail = Option::getOptionsData($prodOptions);
				if(!empty($optionsDetail)) {
					foreach($optionsDetail as $optionDetail) {
						if($optionDetail['option_is_separate_images'] == 1) {
							$optionValObj = new OptionValue();
							$optionValues = $optionValObj->getAtttibutesByOptionId($optionDetail['option_id'], [], $langId);
							
							foreach($optionValues as $optionValue) {
								$optionValueName = $optionValue['optionvalue_identifier']; 
								if(!empty($optionValue['optionvalue_name'])) {
									$optionValueName = $optionValue['optionvalue_name']; 
								}
								$optionValAssArr[$optionValue['optionvalue_id']] = $optionValueName;
							}
						}
					}
				}
			}
		}

		$frm = $this->getImageForm($langId, $optionValAssArr);
        $this->set('frm', $frm);
        $this->set('langId', $langId);
        $this->set('imageSize', imagesSizes::productImageSizeArr()[applicationConstants::getActiveTheme()]);
        $this->set('html', $this->_template->render(false, false, 'products/single-catalog-form/image-form.php', true));
        $this->_template->render(false, false, 'json-success.php', true, false);
	}
	
	private function getImageForm($lang_id = 0, $optionValues = [])
    {
        $frm = new Form('imageFrm');
		$imgTypesArr = array(0 => Labels::getLabel('LBL_For_All_Options', $lang_id));
		
		if(!empty($optionValues)) {
			$imgTypesArr = $imgTypesArr + $optionValues;
		}
		
        $frm->addSelectBox(Labels::getLabel('LBL_IMAGE_FILE_TYPE', $this->adminLangId), 'option_id', $imgTypesArr, 0, array(),'');
        $languagesAssocArr = Language::getAllNames();
        if (count($languagesAssocArr) > 1) {
            $frm->addSelectBox(Labels::getLabel('LBL_LANGUAGE', $this->adminLangId), 'lang_id', array(0 => Labels::getLabel('LBL_All_Languages', $this->adminLangId)) + $languagesAssocArr, '', array(), '');
        } else {
            $langId = array_key_first($languagesAssocArr);
            $frm->addHiddenField('', 'lang_id', $langId);
        }
        $frm->addFileUpload(Labels::getLabel('LBL_UPLOAD', $this->adminLangId), 'prod_image');
        $frm->addHtml('', 'images', '');
        $sizeArr = imagesSizes::productImageSizeArr()[applicationConstants::getActiveTheme()];
        $frm->addHiddenField('', 'min_width',  $sizeArr['width']);
        $frm->addHiddenField('', 'min_height', $sizeArr['height']);
        $frm->addHiddenField('', 'product_id');
        $frm->addHiddenField('', 'file_type');

        return $frm;
    }
	
	public function uploadTempImages()
    {
        $this->objPrivilege->canEditProducts();
        $post = FatApp::getPostedData();
        if (empty($post)) {
            Message::addErrorMessage(Labels::getLabel('LBL_Invalid_Request_Or_File_not_supported', $this->adminLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		$productId = FatApp::getPostedData("product_id", FatUtility::VAR_INT, 0);
		$tempProductId = FatApp::getPostedData("temp_record_id", FatUtility::VAR_INT, 0);
		$optionId = FatApp::getPostedData("option_id", FatUtility::VAR_INT, 0);
		$langId = FatApp::getPostedData("lang_id", FatUtility::VAR_INT, 0);
		
        if (!is_uploaded_file($_FILES['cropped_image']['tmp_name'])) {
            Message::addErrorMessage(Labels::getLabel('LBL_Please_Select_A_File', $this->adminLangId));
            FatUtility::dieJsonError(Message::getHtml());
        }

        if ($_FILES['cropped_image']['size'] > AttachedFile::IMAGE_MAX_SIZE_IN_BYTES_FOR_CROPPER) { /* in kbs */
            Message::addErrorMessage(Labels::getLabel('MSG_Maximum_Upload_Size_is', $this->adminLangId) . ' ' . AttachedFile::IMAGE_MAX_SIZE_IN_BYTES / 1024 . 'KB');
            FatUtility::dieJsonError(Message::getHtml());
        }


        $fileHandlerObj = new AttachedFileTemp();
        if (!$res = $fileHandlerObj->saveImage($_FILES['cropped_image']['tmp_name'], AttachedFile::FILETYPE_PRODUCT_IMAGE_TEMP, $tempProductId, $optionId, $_FILES['cropped_image']['name'], -1, $unique_record = false, $langId)) {
            Message::addErrorMessage($fileHandlerObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }

        $this->set("msg", Labels::getLabel('LBL_Image_Uploaded_Successfully', $this->adminLangId));
        $this->_template->render(false, false, 'json-success.php');
	}
	
	public function productImages()
    {
		$productId = FatApp::getPostedData("product_id", FatUtility::VAR_INT, 0);
		$tempProductId = FatApp::getPostedData("temp_record_id", FatUtility::VAR_INT, 0);
		$optionId = FatApp::getPostedData("option_id", FatUtility::VAR_INT, 0);
		$langId = FatApp::getPostedData("lang_id", FatUtility::VAR_INT, 0);
		$isDefaultLayout = FatApp::getPostedData('isDefaultLayout', FatUtility::VAR_INT, 0);
		
        if ($productId < 1) {
            $productImages = AttachedFileTemp::getMultipleAttachments(AttachedFile::FILETYPE_PRODUCT_IMAGE_TEMP, $tempProductId, $optionId, $langId, false, 0, 0, true);
        } else {
			$productImages = AttachedFile::getMultipleAttachments(AttachedFile::FILETYPE_PRODUCT_IMAGE, $productId, $optionId, $langId, false, 0, 0, true);
		}
		
		if (1 > $langId) {
            $langId = $this->adminLangId;
        }
		
		$this->set('images', $productImages);
        $this->set('recordId', $productId);
        $this->set('langId', $langId);
        $this->set('canEdit', $this->objPrivilege->canEditProducts(0, true));
        $this->set('isDefaultLayout', $isDefaultLayout);
        
        $this->set('html', $this->_template->render(false, false, 'products/single-catalog-form/images.php', true));
        $this->_template->render(false, false, 'json-success.php', true, false);
		
    }
	
	public function deleteProductImage()
    {
        $this->objPrivilege->canEditProducts();
		
		$productId = FatApp::getPostedData("product_id", FatUtility::VAR_INT, 0);
		$tempProductId = FatApp::getPostedData("temp_record_id", FatUtility::VAR_INT, 0);
		$imageId = FatApp::getPostedData("image_id", FatUtility::VAR_INT, 0);
		$isSizeChart = FatApp::getPostedData("isSizeChart", FatUtility::VAR_INT, 0);
		//$isSizeChart = 0; //@todo

        if (($imageId < 1) || ($productId < 1 && $tempProductId < 1)) {
            Message::addErrorMessage($this->str_invalid_request);
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		if($productId > 0) {
			$fileType = AttachedFile::FILETYPE_PRODUCT_IMAGE;
			$fileHandlerObj = new AttachedFile();
		} else {
			$fileType = AttachedFile::FILETYPE_PRODUCT_IMAGE_TEMP;
			$productId = $tempProductId;
			$fileHandlerObj = new AttachedFileTemp();
		}
		
		/*$data = $fileHandlerObj::getAttributesById($imageId, ['afile_lang_id', 'afile_record_subid']);
        if (false == $data) {
            LibHelper::exitWithError($this->str_invalid_request, true);
        }*/
		
        $productObj = new Product();
        if (!$productObj->deleteProductImage($productId, $imageId, $isSizeChart, $fileType)) {
            Message::addErrorMessage($productObj->getError());
            FatUtility::dieJsonError(Message::getHtml());
        }
		
		/*$languages = Language::getAllNames();
        if (count($languages) > 1) {
            $this->set("isDefaultLayout", $data['afile_lang_id'] == 0 && $data['afile_record_subid'] == 0);
        } else {
            $this->set("isDefaultLayout", $data['afile_lang_id'] == CommonHelper::getDefaultFormLangId() && $data['afile_record_subid'] == 0);
        }*/

        $this->set("msg", Labels::getLabel('LBL_Image_Removed_Successfully', $this->adminLangId));
        $this->_template->render(false, false, 'json-success.php');
    }
	
	public function productCatCustomFieldsForm()
    {
        $productId = FatApp::getPostedData('product_id', FatUtility::VAR_INT, 0);
        $catId = FatApp::getPostedData('cat_id', FatUtility::VAR_INT, 0);
        $langId = FatApp::getPostedData('lang_id', FatUtility::VAR_INT, 0);
        if (1 > $catId) {
            $this->set('have_custom_fields', 0);
            $this->_template->render(false, false, 'json-success.php', true, false);
        }

		$updatedProdCatAttr = [];
		
        /*$category = ProductCategory::getAttributesByLangId($langId, $catId);
        if (!is_array($category)) {
            $this->set('have_custom_fields', 0);
            $this->_template->render(false, false, 'json-success.php', true, false);
        }*/
		
        $attrData = ['product_id' => $productId];
		
        if (0 < $productId) {
            
            $numericAttributes = Product::getProductNumericAttributes($productId);
            $textualAttributes = Product::getProductTextualAttributes($productId, $langId);
        
            if (!empty($numericAttributes)) {
                foreach ($numericAttributes as $numericAttribute) {
                    $attrGrpId = $numericAttribute['prodnumattr_attrgrp_id'];
                    unset($numericAttribute['prodnumattr_product_id']);
                    unset($numericAttribute['prodnumattr_attrgrp_id']);
                    $attrData['num_attributes'][$attrGrpId] = $numericAttribute;
                }
            }
            if (!empty($textualAttributes)) {
                foreach ($textualAttributes as $textualAttribute) {
                    $attrGrpId = $textualAttribute['prodtxtattr_attrgrp_id'];
                    unset($textualAttribute['prodtxtattr_product_id']);
                    unset($textualAttribute['prodtxtattr_attrgrp_id']);
                    unset($textualAttribute['prodtxtattr_lang_id']);
                    $attrData['text_attributes'][$attrGrpId][$langId] = $textualAttribute;
                }
            }
        }
        
        $prodCatObj = new ProductCategory($catId);
        $prodCatAttr = $prodCatObj->getAttrDetail($langId);
	
		$siteDefaultLangId = FatApp::getConfig('conf_default_site_lang', FatUtility::VAR_INT, 1);
        foreach ($prodCatAttr as $key => $attr) {
			if(($attr['attr_type'] == AttrGroupAttribute::ATTRTYPE_SELECT_BOX || $attr['attr_type'] == AttrGroupAttribute::ATTRTYPE_CHECKBOXES) && empty($attr['attr_options'])) {
				$attrObj = new AttrGroupAttribute($attr['attr_id']);
				$attrDetails = $attrObj->getDetail();
				if(!empty($attrDetails) && !empty($attrDetails[$siteDefaultLangId]['attr_options'])) {
					$attr['attr_options'] = $prodCatAttr[$key]['attr_options'] = $attrDetails[$siteDefaultLangId]['attr_options'];
				} else if(!empty($attrDetails)) {
					foreach($attrDetails as $attrDetail) {
						if(!empty($attrDetail['attr_options'])) {
							$attr['attr_options'] = $prodCatAttr[$key]['attr_options'] = $attrDetail['attr_options'];
							continue;
						}
					}
				} 
			}
			if(empty($attr['attrlang_lang_id'])){
				$attr['attrlang_lang_id'] = $langId;
			}
            $updatedProdCatAttr[$attr['attr_attrgrp_id']][$attr['attr_id']][$attr['attrlang_lang_id']] = $attr;
        }

		
		$isCustomFields = $prodCatObj->isCategoryHasCustomFields($langId);

        $prod = new Product();
        $prodCatAttr = $prod->formatAttributesData($prodCatAttr);
		
        $frm = $prod->getProdCatCustomFieldsForm($prodCatAttr, $langId, false, $attrData);
        $this->set('frm', $frm);

        $this->set('updatedProdCatAttr', $updatedProdCatAttr);
        $this->set('langId', $langId);
        $this->set('productId', $productId);
        $this->set('html', $this->_template->render(false, false, 'products/single-catalog-form/prod-cat-custom-fields-form.php', true, false));
        $this->set('have_custom_fields', $isCustomFields);
        //$this->set('have_custom_fields', false);
        $this->_template->render(false, false, 'json-success.php', true, false);
    }

    private function setupProdCatCustomFieldsData($prodObj, $post, $langId)
    {
        if (!empty($post['num_attributes'])) {
            for ($i = 1; $i <= 30; $i++) {
                $numeicKeysArr[] = 'prodnumattr_num_' . $i;
            }
            foreach ($post['num_attributes'] as $key => $attributes) {
                $updatedKeys = [];
                foreach ($attributes as $numericKey => $attr) {
                    $num_data_update_arr = array(
                        'prodnumattr_product_id' => $prodObj->getMainTableRecordId(),
                        'prodnumattr_attrgrp_id' => $key,
                        $numericKey => (is_array($attr)) ? implode(',', $attr) : $attr,
                    );
                    $updatedKeys[] = $numericKey;
                    if (!$prodObj->addUpdateNumericAttributes($num_data_update_arr)) {
                        Message::addErrorMessage($prodObj->getError());
                        FatUtility::dieJsonError(Message::getHtml());
                    }
                }
                /* [ UPDATE MISSING FIELDS OR UNCHECKED CHECKBOXES VALUE */
                $missingKeys = array_diff($numeicKeysArr, $updatedKeys);
                if (!empty($missingKeys)) {
                    $num_data_update_arr = array(
                        'prodnumattr_product_id' => $prodObj->getMainTableRecordId(),
                        'prodnumattr_attrgrp_id' => $key
                    );

                    foreach ($missingKeys as $keyName) {
                        $num_data_update_arr[$keyName] = "";
                    }

                    if (!$prodObj->addUpdateNumericAttributes($num_data_update_arr)) {
                        Message::addErrorMessage($prodObj->getError());
                        FatUtility::dieJsonError(Message::getHtml());
                    }
                }
                /* ] */
            }
        }

        if (!empty($post['text_attributes'])) {
            foreach ($post['text_attributes'] as $key => $textAttributes) {
                foreach ($textAttributes as $langId => $attributes) {
                    $text_data_update = array(
                        'prodtxtattr_product_id' => $prodObj->getMainTableRecordId(),
                        'prodtxtattr_attrgrp_id' => $key,
                        'prodtxtattr_lang_id' => $langId,
                    );
                    $text_data_update = array_merge($text_data_update, $attributes);

                    if (!$prodObj->addUpdateTextualAttributes($text_data_update)) {
                        Message::addErrorMessage($prodObj->getError());
                        FatUtility::dieJsonError(Message::getHtml());
                    }
                }
            }
        }
        return true;
    }
	
	public function loadOptionValues()
	{
		$optionValArr = [];
		$post = FatApp::getPostedData();
		$sizeChartCanBeAttached = 0;
		
		if(!empty($post['prod_options'])) {
			$optValObj = new OptionValue();
			foreach($post['prod_options'] as $optionId) {
				$optionValues = $optValObj->getAtttibutesByOptionId($optionId, null, $this->adminLangId);
				if(!empty($optionValues)) {
					foreach($optionValues as $optionVal) {
						$optionValName = $optionVal['optionvalue_identifier'];
						if(!empty($optionVal['optionvalue_name'])) {
							$optionValName = $optionVal['optionvalue_name'];
						}
						$optionValArr[$optionId]['optionValues'][$optionVal['optionvalue_id']] = $optionValName;
					}
				}
			}
			
			if(!empty($optionValArr)) {
				$optionValArr = CommonHelper::combinationOfElementsOfArr($optionValArr, 'optionValues', ' | ');
			}
			
			$optionsDetail = Option::getOptionsData($post['prod_options']);
			if(!empty($optionsDetail)) {
				foreach($optionsDetail as $optionDetail) {
					if($optionDetail['option_attach_sizechart'] == 1) {
						$sizeChartCanBeAttached = 1;
					}
				}
			}
		}
		
		if(!empty($optionValArr)) {
			echo json_encode(['optionValArr' => $optionValArr, 'sizeChartCanBeAttached' => $sizeChartCanBeAttached]);
		} else {
			echo json_encode(['optionValArr' => '', 'sizeChartCanBeAttached' => $sizeChartCanBeAttached]);
		}
		
	}

}
